<?php
require_once __DIR__ . '/../../../init.php';
require_once __DIR__ . '/../../../includes/gatewayfunctions.php';
require_once __DIR__ . '/../../../includes/invoicefunctions.php';

$gatewayModuleName = 'mhpay';
$gatewayParams = getGatewayVariables($gatewayModuleName);
if (!$gatewayParams['type']) {
    die("Module Not Activated");
}

$raw = file_get_contents("php://input");
$data = json_decode($raw, true);

// Optional: if your webhook does NOT send this header, remove this block or replace with your own shared-secret check.
$headers = function_exists('getallheaders') ? getallheaders() : [];
$receivedKey = $headers['mh-mhpay-api-key']
    ?? $headers['Mh-Mhpay-Api-Key']
    ?? $headers['MH-MHPAY-API-KEY']
    ?? '';

if (empty($receivedKey) || $receivedKey !== $gatewayParams['apikey']) {
    header("HTTP/1.1 401 Unauthorized");
    die("Invalid API Key");
}

$invoiceId = isset($data['metadata']['invoiceid']) ? (int)$data['metadata']['invoiceid'] : 0;
$ppId      = $data['pp_id'] ?? '';
$statusRaw = strtolower($data['status'] ?? '');

if (!$invoiceId || !$ppId || !$statusRaw) {
    die("Missing parameters");
}

// WHMCS best practices:
checkCbInvoiceID($invoiceId, $gatewayModuleName); // throws on bad invoice
checkCbTransID($ppId); // prevents double-processing

// Helper to call verify endpoint on a given base URL
$verifyOnce = function (string $base, string $ppId, string $apiKey) {
    $url = rtrim($base, '/') . '/api/verify-payments';
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'Accept: application/json',
            'mh-mhpay-api-key: ' . $apiKey,
        ],
        CURLOPT_POSTFIELDS => json_encode(['pp_id' => $ppId]),
        CURLOPT_TIMEOUT => 30,
    ]);
    $resp = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err  = curl_error($ch);
    curl_close($ch);

    logTransaction($gatewayModuleName, [
        'verify_url' => $url,
        'http' => $http,
        'error' => $err,
        'resp' => $resp,
    ], 'VerifyAttempt');

    $json = json_decode($resp, true);
    return [$http, $json ?: []];
};

// 1) Try primary
list($http1, $ver1) = $verifyOnce($gatewayParams['baseUrl'], $ppId, $gatewayParams['apikey']);

// 2) If not completed, and an alternate URL is configured, try it too
$useAlt = !empty($gatewayParams['alternateUrl']);
$status1 = strtolower($ver1['status'] ?? ($ver1['data']['status'] ?? ''));
if ($status1 !== 'completed' && $useAlt) {
    list($http2, $ver2) = $verifyOnce($gatewayParams['alternateUrl'], $ppId, $gatewayParams['apikey']);
    $ver = $ver2;
    $status = strtolower($ver2['status'] ?? ($ver2['data']['status'] ?? ''));
} else {
    $ver = $ver1;
    $status = $status1;
}

// Pull amount from verification (fallback to webhook)
$amount = $ver['amount'] ?? ($ver['data']['amount'] ?? ($data['amount'] ?? 0));
$amount = is_numeric($amount) ? (float)$amount : 0.0;

// Final decision
if ($status === 'completed') {
    // Add payment (fee optional)
    addInvoicePayment($invoiceId, $ppId, $amount, 0.00, $gatewayModuleName);
    logTransaction($gatewayModuleName, ['pp_id' => $ppId, 'invoice' => $invoiceId, 'amount' => $amount], 'Success');
    http_response_code(200);
    echo "Success";
} else {
    logTransaction($gatewayModuleName, ['verify' => $ver], 'Verification Failed');
    http_response_code(200);
    echo "Payment verification failed.";
}
